﻿using System;
using System.Collections.Generic;
using System.Text;

using Thomas_Erdoesi.Game_Objects;
using Thomas_Erdoesi.Properties;

namespace Thomas_Erdoesi.Game_Analyzer
{
    public class  GameMath
    {
        /// <summary>
        /// Geschwindigkeitsvektor für Abfangkurs berechnen
        /// </summary>
        /// <param name="s">Aktuelle Position des Quellobjekts</param>
        /// <param name="ss">Aktuelle Geschwindigkeit des Quellobjekts</param>
        /// <param name="t">Aktuelle Position des Zielobjekts</param>
        /// <param name="ts">Aktuelle Geschwindigkeit des Zielobjekts</param>
        /// <param name="pv">Geschwindigkeit des Projektils (relativ)</param>
        /// <returns></returns>
        public static GameObjectSpeed CalculateInterceptionSpeed(GameObjectLocation s, GameObjectSpeed ss,
                                                                 GameObjectLocation t, GameObjectSpeed ts, double pv)
        {
            return CalculateInterceptionSpeed(s.x, s.y, ss.sx, ss.sy, t.x, t.y, ts.sx, ts.sy, pv);
        }

        /// <summary>
        /// Geschwindigkeitsvektor für Abfangkurs berechnen
        /// </summary>
        /// <param name="sx">Aktuelle X-Koordinate des Quellobjekts</param>
        /// <param name="sy">Aktuelle Y-Koordinate des Quellobjekts</param>
        /// <param name="ssx">Aktuelle X-Geschwindigkeit des Quellobjekts</param>
        /// <param name="ssy">Aktuelle Y-Geschwindigkeit des Quellobjekts</param>
        /// <param name="tx">Aktuelle X-Koordinate des Zielobjekts</param>
        /// <param name="ty">Aktuelle Y-Koordinate des Zielobjekts</param>
        /// <param name="tsx">Aktuelle X-Geschwindigkeit des Zielobjekts</param>
        /// <param name="tsy">Aktuelle Y-Geschwindigkeit des Zielobjekts</param>
        /// <param name="pv">Geschwindigkeit des Projektils (relativ)</param>
        /// <returns></returns>
        public static GameObjectSpeed CalculateInterceptionSpeed(int sx, int sy, double ssx, double ssy,
                                                                 int tx, int ty, double tsx, double tsy, double pv)
        {
            double dx = tx - sx;                // Distanz Quelle-Ziel in X-Richtung
            double dy = ty - sy;                // Distanz Quelle-Ziel in Y-Richtung

            // Distanz optimieren (Nutzung der Tatsache, dass der Raum endlos ist)
            while (dx > GameConstants.MaxDistX) dx -= GameConstants.SizeX;
            while (dx < GameConstants.MinDistX) dx += GameConstants.SizeX;
            while (dy > GameConstants.MaxDistY) dy -= GameConstants.SizeY;
            while (dy < GameConstants.MinDistY) dy += GameConstants.SizeY;

            // Sicherstellen, dass die Distanz zwischen den Objekten > 0 ist
            if (dx == 0 && dy == 0) return new GameObjectSpeed(0, 0);

            double pv2  = pv * pv;               // Quadrat der Geschwindigkeit des Projektils
            double ssx2 = ssx * ssx;             // Quadrat der X-Geschwindigkeit des Quellobjekts
            double ssy2 = ssy * ssy;             // Quadrat der Y-Geschwindigkeit des Quellobjekts
            double tsx2 = tsx * tsx;             // Quadrat der X-Geschwindigkeit des Zielobjekts
            double tsy2 = tsy * tsy;             // Quadrat der Y-Geschwindigkeit des Zielobjekts

            double h0 = ssx2 - 2 * tsx * ssx + tsx2 + ssy2 - 2 * ssy * tsy + tsy2 - pv2;

            if (h0 != 0)                     // Unterschiedliche Objektgeschwindigkeiten
            {
                // Zeitspanne bis zum Abfangzeitpunkt berechnen
                double dx2 = dx * dx;
                double dy2 = dy * dy;

                double h1 = 2 * (ssx * dx - tsx * dx + ssy * dy - tsy * dy);
                double h2 = Math.Sqrt(h1 * h1 - 4 * h0 * (dx * dx + dy * dy));
                double h3 = 2 * h0;

                double t1 = (h1 - h2) / h3;
                double t2 = (h1 + h2) / h3;
                double t = t1 > t2 ? t1 : t2;

                // Prüfen, ob Abfangen möglich (Zeitspanne muss positiv sein)
                if (t <= 0) return new GameObjectSpeed(0, 0);

                // Geschwindigkeitsvektor für Abfangkurs berechnen
                return new GameObjectSpeed((tsx * t + dx) / t - ssx, (tsy * t + dy) / t - ssy);
            }
            else                                // Identische Objektgeschwindigkeiten
            {
                double h1 = ssx * dx - tsx * dx + ssy * dy - tsy * dy;

                // Prüfen, ob Abfangen möglich
                if (h1 == 0) return new GameObjectSpeed(0, 0);

                // Zeitspanne bis zum Abfangzeitpunkt berechnen
                double dx2 = dx * dx;
                double dy2 = dy * dy;

                double h2 = dx2 + dy2;

                double t = h2 / (2 * h1);

                // Prüfen, ob Abfangen möglich (Zeitspanne muss positiv sein)
                if (t <= 0) return new GameObjectSpeed(0, 0);

                // Abfangkoordinaten berechnen
                double h3 = 2 * dx * dy;

                double psx = (-tsx * dx2 - h3 * tsy + tsx * dy2) / h2;
                double psy = (tsy * dx2 - h3 * tsx - tsy * dy2) / h2;

                // Geschwindigkeitsvektor für Abfangkors berechnen
                return new GameObjectSpeed(psx, psy);
            }
        }

        /// <summary>
        /// Zeit bis zur Kollision berechnen
        /// </summary>
        /// <param name="s">Aktuelle Koordinaten des Quellobjekts</param>
        /// <param name="ss">Aktuelle Geschwindigkeit des Quellobjekts</param>
        /// <param name="sr">Radius des Quellobjekts</param>
        /// <param name="t">Aktuelle Koordinaten des Zielobjekts</param>
        /// <param name="ts">Aktuelle Geschwindigkeit des Zielobjekts</param>
        /// <param name="tr">Radius des Zielobjekts</param>
        /// <returns></returns>
        public static double CalculateTimeToCollision(GameObjectLocation s, GameObjectSpeed ss, double sr,
                                                      GameObjectLocation t, GameObjectSpeed ts, double tr)
        {
            return CalculateTimeToCollision(s.x, s.y, ss.sx, ss.sy, sr, t.x, t.y, ts.sx, ts.sy, tr);
        }

        /// <summary>
        /// Zeit bis zur Kollision berechnen
        /// </summary>
        /// <param name="sx">Aktuelle X-Koordinate des Quellobjekts</param>
        /// <param name="sy">Aktuelle Y-Koordinate des Quellobjekts</param>
        /// <param name="ssx">Aktuelle X-Geschwindigkeit des Quellobjekts</param>
        /// <param name="ssy">Aktuelle Y-Geschwindigkeit des Quellobjekts</param>
        /// <param name="sr">Radius des Quellobjekts</param>
        /// <param name="tx">Aktuelle X-Koordinate des Zielobjekts</param>
        /// <param name="ty">Aktuelle Y-Koordinate des Zielobjekts</param>
        /// <param name="tsx">Aktuelle X-Geschwindigkeit des Zielobjekts</param>
        /// <param name="tsy">Aktuelle Y-Geschwindigkeit des Zielobjekts</param>
        /// <param name="tr">Radius des Zielobjekts</param>
        /// <returns></returns>
        public static double CalculateTimeToCollision(double sx, double sy, double ssx, double ssy, double sr,
                                                      double tx, double ty, double tsx, double tsy, double tr)
        {
            double dmin = sr + tr;
            double dx = tx - sx;
            double dy = ty - sy;                
            double dsx = tsx - ssx;
            double dsy = tsy - ssy;

            double Result = double.MaxValue;

            for (double ddy = -GameConstants.SizeY; ddy <= GameConstants.SizeY; ddy +=GameConstants.SizeY)
                for (double ddx = -GameConstants.SizeX; ddx <= GameConstants.SizeX; ddx += GameConstants.SizeX)
                    Result = Math.Min(Result, CalculateTimeToCollision(dx + ddx, dy + ddy, dsx, dsy, dmin));

            return Result;
        }

        public static double CalculateTimeToCollision(double dx, double dy, double dsx, double dsy, double dmin)
        {
            // Objekte sind bereits kollidiert
            if (dx * dx + dy * dy <= dmin * dmin) return 0;

            // Prüfen, ob einen Kollision zu erwarten ist
            double dsx2_plus_dsy2 = dsx * dsx + dsy * dsy;

            if (dsx2_plus_dsy2 == 0) return double.MaxValue;


            double dsy_dx_minus_dsx_dy = dsy * dx - dsx * dy;

            double w1 = dmin * dmin * dsx2_plus_dsy2 - dsy_dx_minus_dsx_dy * dsy_dx_minus_dsx_dy;

            if (w1 < 0) return double.MaxValue;

            // Kollisionszeitpunkt berechnen
            double w2 = Math.Sqrt(w1);

            double t1 = (-dsx * dx - dsy * dy - w2) / dsx2_plus_dsy2;
            double t2 = (-dsx * dx - dsy * dy + w2) / dsx2_plus_dsy2;

            // Auseinanderbewegung (Kollision lag in der Vergangenheit)
            if (t1 < 0 && t2 < 0) return double.MaxValue;

            // Zeit bis zu Kollision zurückliefern
            return Math.Min(t1, t2);
        }

        /// <summary>
        /// Geschwindigkeit eines eigenen Schusses berechnen
        /// </summary>
        /// <param name="Spaceship"></param>
        /// <returns></returns>
        public static GameObjectSpeed CalculateShotSpeed(GameObjectSpeed SpaceshipSpeed, double SpaceshipAngle)
        {
            double ShotSpeedX = SpaceshipSpeed.sx + Math.Cos(SpaceshipAngle) * Settings.Default.EstimatedShotSpeed;
            double ShotSpeedY = SpaceshipSpeed.sy + Math.Sin(SpaceshipAngle) * Settings.Default.EstimatedShotSpeed;

            GameObjectSpeed Result = new GameObjectSpeed(ShotSpeedX, ShotSpeedY);

            if (Result.sx > Settings.Default.MaxShotSpeed) Result.sx = Settings.Default.MaxShotSpeed;
            if (Result.sy > Settings.Default.MaxShotSpeed) Result.sy = Settings.Default.MaxShotSpeed;

            return Result;
        }

        /// <summary>
        /// Differenz zwischen zwei Winkeln berechnen
        /// </summary>
        /// <param name="Angle1"></param>
        /// <param name="Angle2"></param>
        /// <returns></returns>
        public static double AngleDiff(double Angle1, double Angle2)
        {
            double Result = Angle2 - Angle1;

            if (Result > Math.PI) Result -= 2 * Math.PI;
            if (Result < -Math.PI) Result += 2 * Math.PI;

            return Result;
        }

        public static GameObjectLocation CalculateInitialShotLocation(GameObjectLocation SpaceshipLocation, double SpaceshipAngle)
        {
            return new GameObjectLocation((int)(SpaceshipLocation.x + Settings.Default.InitialShotDistance * Math.Cos(SpaceshipAngle)),
                                          (int)(SpaceshipLocation.y + Settings.Default.InitialShotDistance * Math.Sin(SpaceshipAngle)));
        }
    }
}
